#include <device.h>

static const uint8 SLAVE_ADDRESS = 0x3e;

/* --- send buffer */
static unsigned char lcd_send_buf[2];

/* HEX char */
const unsigned char cHexCode[16] = {
  '0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'
};

uint8 fmLcdWaitComplete(void)
{
	uint8 result = SCB_1_I2C_MSTAT_XFER_INP;

	while(result == SCB_1_I2C_MSTAT_XFER_INP) {
		result = SCB_1_I2CMasterStatus();
	}

	if ((result == SCB_1_I2C_MSTAT_RD_CMPLT) ||
		(result == SCB_1_I2C_MSTAT_WR_CMPLT)) {
		result = 0;
	}
	return result;
}


void fmLcdSendCommand(uint8 com)
{
	lcd_send_buf[0] = 0x00;
	lcd_send_buf[1] = com;
	
	SCB_1_I2CMasterClearStatus();
	SCB_1_I2CMasterWriteBuf(SLAVE_ADDRESS, lcd_send_buf, 2, SCB_1_I2C_MODE_COMPLETE_XFER);
	fmLcdWaitComplete();
	CyDelayUs(40);
}


void fmLcdPutChar(uint8 chara)
{
	lcd_send_buf[0] = 0x40;
	lcd_send_buf[1] = chara;
	
	SCB_1_I2CMasterClearStatus();
	SCB_1_I2CMasterWriteBuf(SLAVE_ADDRESS, lcd_send_buf, 2, SCB_1_I2C_MODE_COMPLETE_XFER);
	fmLcdWaitComplete();
	CyDelayUs(40);
}


void fmLcdClear(void)
{
	fmLcdSendCommand(0x01);
	CyDelayUs(2000);
}


void fmLcdLocate(uint8 pos_x, uint8 pos_y)
{
	uint8 pos = (pos_y * 0x40 + pos_x) | 0x80;
	fmLcdSendCommand(pos);
}


void fmLcdPutMessage(uint8 *chara, uint8 length)
{
	uint8 loop;
	for(loop=0; loop<length; loop++) {
		fmLcdPutChar(chara[loop]);
	}
}


/* ------------------------------- */
/* --- initial code */
void fmLcdInitialize(void)
{
	CyDelayUs(10000);
	CyDelayUs(10000);
	CyDelayUs(10000);
	CyDelayUs(10000);
	CyDelayUs(10000);
	fmLcdSendCommand(0x38);
	CyDelayUs(1000);
	fmLcdSendCommand(0x39);
	CyDelayUs(1000);
	fmLcdSendCommand(0x14);
	CyDelayUs(1000);
	fmLcdSendCommand(0x7a);
	CyDelayUs(1000);
	fmLcdSendCommand(0x5e);
	CyDelayUs(1000);
	fmLcdSendCommand(0x6a);
	CyDelayUs(1000);
	fmLcdSendCommand(0x0c);
	CyDelayUs(1000);
	fmLcdClear();
    fmLcdLocate(0,0);
}


/* ------------------------------- */
/* --- display HEX value */
void fmLcdDisplayHex_8(const uint8 val)
{
  unsigned char c_low;

  c_low = cHexCode[((val & 0xf0) >> 4)];
  fmLcdPutChar(c_low);
  c_low = cHexCode[(val & 0x0f)];
  fmLcdPutChar(c_low);
}

void fmLcdDisplayHex_16(const uint16 val)
{
  unsigned char c_low;

  c_low = (val & 0xff00) >> 8;
  fmLcdDisplayHex_8(c_low);
  c_low = val & 0x00ff;
  fmLcdDisplayHex_8(c_low);
}

void fmLcdDisplayHex_32(const uint32 val)
{
  uint16 c_low;

  c_low = (val & 0xffff0000) >> 16;
  fmLcdDisplayHex_16(c_low);
  c_low = val & 0x0000ffff;
  fmLcdDisplayHex_16(c_low);
}


/* ------------------------------- */
/* --- display DECIMAL value */
void fmLcdDisplayDec_24(const uint32 num)
{
	uint32 val, t;

    val = num;
	if (val & 0x800000) {
		fmLcdPutChar('-');
		val &= 0x7fffff;
	}
	else {
		fmLcdPutChar(' ');
	}
	t = val / 100000;
	fmLcdPutChar(cHexCode[t]);
	t *= 100000;
	val -= t;
	t = val / 10000;
	fmLcdPutChar(cHexCode[t]);
	t *= 10000;
	val -= t;
	t = val / 1000;
	fmLcdPutChar(cHexCode[t]);
	t *= 1000;
	val -= t;
	t = val / 100;
	fmLcdPutChar(cHexCode[t]);
	t *= 100;
	val -= t;
	t = val / 10;
	fmLcdPutChar(cHexCode[t]);
	t *= 10;
	val -= t;
	fmLcdPutChar(cHexCode[val]);
}

void fmLcdDisplayDec_16(const uint16 num)
{
	uint16 val, t;

    val = num;
	t = val / 10000;
	fmLcdPutChar(cHexCode[t]);
	t *= 10000;
	val -= t;
	t = val / 1000;
	fmLcdPutChar(cHexCode[t]);
	t *= 1000;
	val -= t;
	t = val / 100;
	fmLcdPutChar(cHexCode[t]);
	t *= 100;
	val -= t;
	t = val / 10;
	fmLcdPutChar(cHexCode[t]);
	t *= 10;
	val -= t;
	fmLcdPutChar(cHexCode[val]);
}

void fmLcdDisplayDec_8(const uint8 num)
{
	uint8 val, t;

    val = num;
	t = val / 100;
	fmLcdPutChar(cHexCode[t]);
	t *= 100;
	val -= t;
	t = val / 10;
	fmLcdPutChar(cHexCode[t]);
	t *= 10;
	val -= t;
	fmLcdPutChar(cHexCode[val]);
}
